<?php
/**
 * Final System Verification
 * Complete verification of the VIP-based rebate system
 */

require_once("../../conn.php");
require_once("rebate_functions.php");

date_default_timezone_set('Asia/Kolkata');

echo "<h2>🎯 Final System Verification</h2>";

// System Configuration Display
echo "<div style='background: #e7f3ff; padding: 15px; border-radius: 5px; color: #004085; margin-bottom: 20px;'>";
echo "<h3>📋 System Configuration</h3>";
echo "<strong>Database Column Mappings:</strong><br>";
echo "• User ID (VIP Table): <code>userid</code><br>";
echo "• VIP Level (VIP Table): <code>lvl</code><br>";
echo "• User ID (Wallet Table): <code>balakedara</code><br>";
echo "• Wallet Balance (Wallet Table): <code>motta</code><br>";
echo "• User ID (Rebate Table): <code>user_id</code><br>";
echo "• Payment Tracking: <code>is_paid</code>, <code>paid_at</code><br>";
echo "</div>";

// VIP Rate Configuration
echo "<div style='background: #fff3cd; padding: 15px; border-radius: 5px; color: #856404; margin-bottom: 20px;'>";
echo "<h3>💎 VIP Rate Configuration</h3>";
$vipRates = [
    0 => 0.0000, 1 => 0.0005, 2 => 0.0008, 3 => 0.0012, 4 => 0.0015,
    5 => 0.0018, 6 => 0.0020, 7 => 0.0022, 8 => 0.0025, 9 => 0.0028, 10 => 0.0030
];

echo "<table border='1' style='border-collapse: collapse; width: 100%; background: white;'>";
echo "<tr style='background: #ffc107; color: #212529;'><th>VIP Level</th><th>Rebate Rate</th><th>Example on ₹1000 bet</th></tr>";
foreach ($vipRates as $level => $rate) {
    $example = $rate * 1000;
    echo "<tr>";
    echo "<td>VIP $level</td>";
    echo "<td>" . ($rate * 100) . "%</td>";
    echo "<td>₹" . number_format($example, 2) . "</td>";
    echo "</tr>";
}
echo "</table>";
echo "</div>";

// Critical Tests
$criticalTests = [
    'database_connection' => false,
    'table_existence' => false,
    'column_mapping' => false,
    'vip_rates' => false,
    'wallet_operations' => false,
    'rebate_processing' => false
];

// Test 1: Database Connection
echo "<h3>🔗 Test 1: Database Connection</h3>";
if ($conn->connect_error) {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>❌ FAIL:</strong> " . $conn->connect_error;
    echo "</div>";
} else {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✅ PASS:</strong> Database connection successful";
    echo "</div>";
    $criticalTests['database_connection'] = true;
}

// Test 2: Table Existence
echo "<h3>📋 Test 2: Table Existence</h3>";
$requiredTables = ['bet_rebates', 'vip', 'shonu_kaichila'];
$tableResults = [];

foreach ($requiredTables as $table) {
    $tableCheck = $conn->query("SHOW TABLES LIKE '$table'");
    $exists = ($tableCheck && $tableCheck->num_rows > 0);
    $tableResults[$table] = $exists;
    
    if ($exists) {
        echo "<span style='color: green;'>✅ $table</span> ";
    } else {
        echo "<span style='color: red;'>❌ $table</span> ";
    }
}

$criticalTests['table_existence'] = !in_array(false, $tableResults);
echo "<br><br>";

// Test 3: Column Mapping Verification
echo "<h3>🗂️ Test 3: Column Mapping Verification</h3>";
$columnTests = [
    'vip.userid' => "SELECT userid FROM vip LIMIT 1",
    'vip.lvl' => "SELECT lvl FROM vip LIMIT 1",
    'shonu_kaichila.balakedara' => "SELECT balakedara FROM shonu_kaichila LIMIT 1",
    'shonu_kaichila.motta' => "SELECT motta FROM shonu_kaichila LIMIT 1",
    'bet_rebates.user_id' => "SELECT user_id FROM bet_rebates LIMIT 1",
    'bet_rebates.is_paid' => "SELECT is_paid FROM bet_rebates LIMIT 1"
];

$columnResults = [];
foreach ($columnTests as $column => $query) {
    $result = $conn->query($query);
    $exists = ($result !== false);
    $columnResults[$column] = $exists;
    
    if ($exists) {
        echo "<span style='color: green;'>✅ $column</span><br>";
    } else {
        echo "<span style='color: red;'>❌ $column</span><br>";
    }
}

$criticalTests['column_mapping'] = !in_array(false, $columnResults);

// Test 4: VIP Rate Functions
echo "<h3>💎 Test 4: VIP Rate Functions</h3>";
$rateTestResults = [];
foreach ($vipRates as $level => $expectedRate) {
    $actualRate = getVipRebateRate($level);
    $correct = (abs($actualRate - $expectedRate) < 0.0001);
    $rateTestResults[$level] = $correct;
    
    if ($correct) {
        echo "<span style='color: green;'>✅ VIP $level: " . ($actualRate * 100) . "%</span> ";
    } else {
        echo "<span style='color: red;'>❌ VIP $level: Expected " . ($expectedRate * 100) . "%, Got " . ($actualRate * 100) . "%</span> ";
    }
}

$criticalTests['vip_rates'] = !in_array(false, $rateTestResults);
echo "<br><br>";

// Test 5: Wallet Operations
echo "<h3>💰 Test 5: Wallet Operations</h3>";
$walletTestQuery = "SELECT sk.balakedara as user_id, sk.motta as balance 
                   FROM shonu_kaichila sk 
                   WHERE sk.motta > 0 
                   ORDER BY sk.motta DESC 
                   LIMIT 3";

$walletResult = $conn->query($walletTestQuery);
if ($walletResult && $walletResult->num_rows > 0) {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✅ PASS:</strong> Wallet operations working";
    echo "</div>";
    
    echo "<table border='1' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
    echo "<tr style='background: #f2f2f2;'><th>User ID</th><th>Current Balance</th></tr>";
    while ($row = $walletResult->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($row['user_id']) . "</td>";
        echo "<td>₹" . number_format($row['balance'], 2) . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    $criticalTests['wallet_operations'] = true;
} else {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>❌ FAIL:</strong> Wallet operations not working";
    echo "</div>";
}

// Test 6: Rebate Processing Simulation
echo "<h3>🔄 Test 6: Rebate Processing Simulation</h3>";
$rebateTestQuery = "SELECT br.user_id, br.rebate_amount, br.vip_level, sk.motta as current_balance,
                          (sk.motta + br.rebate_amount) as new_balance
                   FROM bet_rebates br
                   JOIN shonu_kaichila sk ON br.user_id = sk.balakedara
                   WHERE (br.is_paid IS NULL OR br.is_paid = 0)
                   ORDER BY br.rebate_amount DESC
                   LIMIT 3";

$rebateResult = $conn->query($rebateTestQuery);
if ($rebateResult && $rebateResult->num_rows > 0) {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✅ PASS:</strong> Rebate processing simulation successful";
    echo "</div>";
    
    echo "<table border='1' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
    echo "<tr style='background: #f2f2f2;'><th>User ID</th><th>Rebate Amount</th><th>VIP Level</th><th>Current Balance</th><th>New Balance</th></tr>";
    while ($row = $rebateResult->fetch_assoc()) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($row['user_id']) . "</td>";
        echo "<td>₹" . number_format($row['rebate_amount'], 2) . "</td>";
        echo "<td>" . htmlspecialchars($row['vip_level']) . "</td>";
        echo "<td>₹" . number_format($row['current_balance'], 2) . "</td>";
        echo "<td>₹" . number_format($row['new_balance'], 2) . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    $criticalTests['rebate_processing'] = true;
} else {
    echo "<div style='background: #fff3cd; padding: 10px; border-radius: 5px; color: #856404;'>";
    echo "<strong>ℹ️ INFO:</strong> No unpaid rebates found to simulate";
    echo "</div>";
    $criticalTests['rebate_processing'] = true; // Consider this as pass since no data doesn't mean failure
}

// Final System Status
echo "<h3>🎯 Final System Status</h3>";
$allTestsPassed = !in_array(false, $criticalTests);

if ($allTestsPassed) {
    echo "<div style='background: #d4edda; padding: 20px; border-radius: 10px; color: #155724; border: 3px solid #28a745;'>";
    echo "<h2 style='color: #155724; margin-top: 0;'>🎉 SYSTEM FULLY OPERATIONAL!</h2>";
    echo "<strong>Your VIP-based Daily Rebate System is ready for production!</strong><br><br>";
    
    echo "<strong>✅ All Critical Tests Passed:</strong><br>";
    echo "• Database Connection: Working<br>";
    echo "• Table Structure: Complete<br>";
    echo "• Column Mappings: Verified<br>";
    echo "• VIP Rate Functions: All 11 levels working<br>";
    echo "• Wallet Operations: Functional<br>";
    echo "• Rebate Processing: Ready<br><br>";
    
    echo "<strong>🚀 Next Steps:</strong><br>";
    echo "1. <strong>Set up cron job:</strong> Schedule daily_rebate_payout_cron.php to run at 12:01 AM<br>";
    echo "2. <strong>Test manually:</strong> Use test_rebate_cron.php for testing<br>";
    echo "3. <strong>Monitor logs:</strong> Check rebate_cron.log for execution status<br>";
    echo "4. <strong>Production deployment:</strong> Start with a test period<br><br>";
    
    echo "<strong>📊 System Capabilities:</strong><br>";
    echo "• VIP Level 0: 0% rebate (No VIP)<br>";
    echo "• VIP Level 1-10: 0.05% to 0.30% rebate<br>";
    echo "• Daily automatic processing at 12:01 AM<br>";
    echo "• Duplicate payment prevention<br>";
    echo "• Comprehensive logging and monitoring<br>";
    echo "• Safe transaction handling<br>";
    echo "</div>";
} else {
    echo "<div style='background: #f8d7da; padding: 20px; border-radius: 10px; color: #721c24; border: 3px solid #dc3545;'>";
    echo "<h2 style='color: #721c24; margin-top: 0;'>❌ SYSTEM NEEDS ATTENTION!</h2>";
    echo "<strong>Some critical tests failed. Please review the issues above.</strong><br><br>";
    
    echo "<strong>Failed Tests:</strong><br>";
    foreach ($criticalTests as $test => $passed) {
        if (!$passed) {
            echo "• " . str_replace('_', ' ', ucfirst($test)) . "<br>";
        }
    }
    
    echo "<br><strong>Recommended Actions:</strong><br>";
    echo "1. Check database connectivity and permissions<br>";
    echo "2. Verify all required tables exist<br>";
    echo "3. Run database migration scripts if needed<br>";
    echo "4. Contact system administrator for assistance<br>";
    echo "</div>";
}

// Usage Statistics
echo "<h3>📈 Current System Statistics</h3>";
$statsQuery = "SELECT 
                  COUNT(*) as total_rebates,
                  SUM(rebate_amount) as total_amount,
                  COUNT(DISTINCT user_id) as unique_users,
                  SUM(CASE WHEN is_paid = 1 THEN rebate_amount ELSE 0 END) as paid_amount,
                  SUM(CASE WHEN is_paid = 0 OR is_paid IS NULL THEN rebate_amount ELSE 0 END) as pending_amount
                FROM bet_rebates";

$statsResult = $conn->query($statsQuery);
if ($statsResult) {
    $stats = $statsResult->fetch_assoc();
    echo "<div style='background: #e7f3ff; padding: 15px; border-radius: 5px; color: #004085;'>";
    echo "<strong>📊 System Statistics:</strong><br>";
    echo "• Total Rebates Generated: " . number_format($stats['total_rebates']) . "<br>";
    echo "• Total Rebate Amount: ₹" . number_format($stats['total_amount'], 2) . "<br>";
    echo "• Unique Users: " . number_format($stats['unique_users']) . "<br>";
    echo "• Paid Amount: ₹" . number_format($stats['paid_amount'], 2) . "<br>";
    echo "• Pending Amount: ₹" . number_format($stats['pending_amount'], 2) . "<br>";
    echo "</div>";
}

$conn->close();
?>

<style>
body {
    font-family: Arial, sans-serif;
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
    background-color: #f8f9fa;
}

h2 {
    color: #333;
    border-bottom: 2px solid #007bff;
    padding-bottom: 10px;
}

h3 {
    color: #555;
    margin-top: 30px;
    margin-bottom: 15px;
}

code {
    background-color: #f8f9fa;
    padding: 2px 4px;
    border-radius: 3px;
    font-family: 'Courier New', monospace;
    color: #d63384;
}

table {
    margin: 10px 0;
    background: white;
    border-radius: 5px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

th {
    background-color: #007bff;
    color: white;
    padding: 12px;
    text-align: left;
}

td {
    padding: 8px 12px;
    border-bottom: 1px solid #eee;
}

tr:hover {
    background-color: #f8f9fa;
}
</style>
